#ifndef _tsnode_h
#define _tsnode_h

// $Id: tsnode.h,v 1.5 1998/11/27 20:09:48 oliva Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "node.h"
#include "visit_node.h"
#include "adjlist.h"
#include "arc.h"

/** Implements the basic state machine of a node for the Topological
    Sort algorithm.  */
class tsnode_state_type {
  /** The `unreached' state means the node was not reached yet, so its
      distance is not meaningful.  `unreached_labeled' is mostly like
      `unreached', but the node is already active.  This can happen in
      the `gor' algorithm.  `reached_labeled' means the distance of
      the node was recently updated, but it was not scanned for
      shorter paths to children yet.  `scanned' means the arcs leaving
      the node were scanned since its last update. */
  enum { unreached, unreached_labeled, reached_labeled, scanned } status;

public:
  /** Initializes the status as unreached.  */
  tsnode_state_type() : status(unreached) {}

  /** Returns false if the current state is unreached or unreached_labeled.  */
  bool is_reached() const { return status > unreached_labeled; }
  
  /** Returns true unless the current state is reached_labeled or
      unreached_labeled.  */
  bool is_labeled() const {
    /* we hope this is faster than
       status == unreached_labeled || status == reached_labeled */
    return (unsigned(status-1)) <= unsigned(reached_labeled-1);
  }

  /** If node was already reached, or is being forced to be reached,
      mark it as reached_labeled.  Otherwise, mark it as
      unreached_labeled.  */
  void mark_labeled(bool force_reached = true) {
    if (force_reached || is_reached())
      status = reached_labeled;
    else
      status = unreached_labeled;
  }

  /** Enter scanned state.  */
  void mark_scanned() { status = scanned; }
};

/** This is the node type used in the Topological Scan algorithm.  */
class tsnode : public visit_node,
	       public node<tsnode, adjlist<arc<tsnode> >, tsnode_state_type>
{
 private:
  /** Just a shorthand to refer to the base class.  */
  typedef node<tsnode, adjlist<arc<tsnode> >, tsnode_state_type> inherited;
    
 public:
  /* Unlike the implementation in the base class, this method won't
     force the node to become reached.  Note that
     inherited::set_source and inherited::test_relabel, will NOT call
     this specialized version. */
  void mark_labeled() {
    inherited::status.mark_labeled(false);
  }

 public:
  /** Just forward the identified to the base class.  */
  tsnode(node_id_t id) : inherited(id) {}
};

#endif
