// $Id: pape.cc,v 1.6 1998/11/29 18:38:46 islene Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Pape-Levit's algorithm: This algorithm [GoldbergRadzik93] uses a
 * deque (a queue that allows insertions at either end) to maintain
 * labeled nodes. The next node to be scanned is removed from the head
 * of the queue. A node that becomes labeled is added to the tail of
 * the queue if this is the first time this node became labeled, and
 * to the head of the queue otherwise (i.e., if it has been scanned
 * previously).
 */
 
#include "spalgo.h"
#include "graph.h"
#include "node.h"
#include "basic_node.h"

#include <deque>

class pape_levit :
  public shortest_path_algorithm<pape_levit, graph< basic_node > > {
  /** Convenience type alias.  */
  typedef shortest_path_algorithm<pape_levit, graph< basic_node > > inherited;

  /** Double-ended queue of nodes to scan.  */
  std::deque<node_t*> to_scan;
  
  /** Inserts the node in the front or in the back of the queue,
      depending on whether it had already been reached or not.  */
  void push(node_t& node, bool was_reached = true) {
    if (was_reached)
      to_scan.push_front(&node); // scan it sooner
    else
      to_scan.push_back(&node); // scan it later
  }

  /** Returns true if there are no more nodes left to scan.  */
  bool empty() const {
    return to_scan.empty();
  }

  /** Removes a node from the front of the queue.  */
  node_t *pop() {
    node_t *node = to_scan.front();
    to_scan.pop_front();
    return node;
  }

public:
  /** Initialize the queue with the source node.  */
  pape_levit(graph_t& g) : inherited(g) {
    push(g.source()); // scan source first
  }

  /** If queue is empty, return NULL, otherwise, pop() the first
      element.  */
  node_t *select() {
    if (empty())
      return 0;
    return pop();
  }

  /** Return true iff the node was relabeled.  If it has become
      labeled, it is inserted in the queue.  */
  bool test_relabel(arc_t& arc) {
    bool was_labeled = arc.to().is_labeled();
    bool was_reached = arc.to().is_reached();
    bool relabeled = inherited::test_relabel(arc);
    // was_labeled implies it was in the queue alread
    if (relabeled && !was_labeled)
      push(arc.to(), was_reached); // scan it, sooner or later
    return relabeled;
  }
};

typedef pape_levit algorithm_t;

#include "main.h"
