#ifndef _in2bin_node_h
#define _in2bin_node_h

// $Id: in2bin_node.h,v 1.2 1998/11/29 01:28:18 oliva Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "node.h"
#include "adjlist.h"
#include "arc.h"

#include <stack>

/** This is a class to isolate the implementation of the delayed arc
    queue from the implementation of the in2bin_node.  */
template <class arc_t> class in2bin_container {
  /** This is the type of container we're going to use */
  typedef std::stack<arc_t*> queue_t;

  /** If we're using a deque to implement the queue (which stack
      does), egcs 1.1 would crash in some circumstances.  */
#ifndef QUEUE_ASSIGN_WITH_RECONSTRUCTION
#define QUEUE_ASSIGN_WITH_RECONSTRUCTION 1
#endif

  /** Should we hold a pointer to the actual container (1) or contain
      an object of the container type (0)?  The performance tests have
      shown that direct containment is only slightly faster.  However,
      it may be possible to save some memory by using indirect
      containment with on-demand allocation.  */
#ifndef QUEUE_BY_PTR
#define QUEUE_BY_PTR 0
#endif

#if QUEUE_BY_PTR
  /** Using a pointer allows us to dynamically allocate and
      disallocate queues, which saves us memory and, surprisingly
      again, does not compromise performance.  */
#ifndef QUEUE_BY_PTR_ON_DEMAND
#define QUEUE_BY_PTR_ON_DEMAND 1
#endif

  /** Since we're supposed to use pointers, declare one.  */
  queue_t *queue_;

  /** Just to arrange that other member functions don't have to care
      which representation we're using.  Create the queue on demand.  */
  queue_t &queue() {
# if QUEUE_BY_PTR_ON_DEMAND
    if (!queue_)
      queue_ = new queue_t;
# endif
    return *queue_;
  }

  /** Check whether the queue is empty.  If we're doing on-demand
      allocation, don't allocate it, and, if it becomes empty,
      de-allocate it.  */
  bool queue_empty() {
# if QUEUE_BY_PTR_ON_DEMAND
    if (!queue_)
      return true;

    if (queue_->empty()) {
      delete queue_;
      queue_ = 0;
      return true;
    } else
      return false;
# else
    return queue_->empty();
# endif
  }

 public:
  /** If we're not doing on-demand allocation, allocate the queue here.  */
  in2bin_container() : queue_(QUEUE_BY_PTR_ON_DEMAND ? 0 : new queue_t()) {}

  /** Destruct the queue.  Even if we were allocating on demand, this
      can't hurt.  */
  ~in2bin_container() { delete queue_; }
  
  /** Since the destructor relies on the fact that we have only one
      node pointing to each queue, we have to copy-construct it here.  */
  in2bin_container(const in2bin_container& other) :
    queue_(QUEUE_BY_PTR_ON_DEMAND && !other.queue_ ? 0 :
	   new queue_t(*other.queue_)) {}

  /** Since the destructor relies on the fact that we have only one
      node pointing to each queue, we have to copy it here.  */
  in2bin_container& operator=(const in2bin_container& other) {
    if (!QUEUE_BY_PTR_ON_DEMAND || (queue_ && other.queue_)) {
# if QUEUE_ASSIGN_WITH_RECONSTRUCTION
      delete queue_;
      queue_ = new queue_t(*other.queue_);
# else
      *queue_ = *other.queue_;
# endif
    }
# if QUEUE_BY_PTR_ON_DEMAND
    else {
      delete queue_;
      queue_ = (QUEUE_BY_PTR_ON_DEMAND && !other.queue_ ? 0 :
		new queue_t(*other.queue_));
    }
# endif
    return *this;
  }
	   
#else /* from now on, QUEUE_BY_PTR == 0 */

  /** The queue is now part of the object.  */
  queue_t queue_;

  /** Just return the queue.  */
  queue_t &queue() { return queue_; }

  /** Check whether the queue is empty.  */
  bool queue_empty() { return queue_.empty(); }

 public:
# if QUEUE_ASSIGN_WITH_RECONSTRUCTION
  /** Ugh.  egcs 1.1 would crash if we would use the default
      assignment operator of deque here.  Calling it explicitly
      wouldn't help either.  Anyway, destructing the current one and
      creating a new one is likely to be equivalent.  */
  in2bin_container& operator=(in2bin_container const& other) {
    queue_.~queue_t();
    new (&queue_) queue_t(other.queue_);
  }
# endif

#endif /* QUEUE_BY_PTR */

  /** Forward the push request to the actual container.  */
  void push(arc_t& arc) {
    queue().push(&arc);
  }

  /** If the queue is empty, return NULL.  Otherwise, return the top
      node and pop it.  */
  arc_t *pop() {
    if (queue_empty())
      return 0;

    arc_t *arc = queue().top();
    queue().pop();
    return arc;
  }
};

/** We don't need any states for this node.  */
class in2bin_state_type {};

class in2bin_node :
  public node<in2bin_node, adjlist<arc<in2bin_node> >, in2bin_state_type> {
 private:
  /** Just a shorthand to refer to the base class.  */
  typedef node<in2bin_node, adjlist<arc<in2bin_node> >, in2bin_state_type> inherited;

  typedef in2bin_container<arc_t> arc_queue_t;

  /** A queue of arcs whose processing was delayed.  */
  arc_queue_t delayed;

 public:
  /** Enqueue an arc for later processing.  If the queue was not
      created yet, create it now.  */
  void delay(arc_t &arc) { delayed.push(arc); }

  /** Get next arc.  */
  arc_t *next_delayed_arc() { return delayed.pop(); }

 public:
  /** Just forward the identified to the base class.  */
  in2bin_node(inherited::node_id_t id) : inherited(id) {}
};

#endif

