// $Id: gor.cc,v 1.8 1998/11/29 18:38:45 islene Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Topological-Scan (GOR): This algorithm [GoldbergRadzik93] is a
 * generalization of the parent-checking idea. Initiallqy, assume that
 * G has no cycles of length zero or less, and therefore, G_d is
 * acyclic.  The topological-scan algorithm maintains the set of
 * labeled nodes in two sets, A and B.  At the beginning of each pass,
 * the algorithm uses the set B to compute the set A of nodes to be
 * scanned during the pass, and resets B to the empty set. A is a
 * linearly ordered set. During the pass, elements are removed
 * according to the ordering of A and scanned.  The newly created
 * labeled nodes are added to B. The algorithm terminates when B is
 * empty at the end of a pass. The algorithm computes A from B as
 * follows:
 * 
 *  - For every v in B that has no outgoing arc with negative reduced
 *  cost, delete v from B and mark it as scanned.
 *     
 *  - Let A be the set of nodes reachable from B in Gd. Mark all nodes
 *  in A as labeled.
 *
 *  - Apply topological sort to A so that for every pair of nodes v
 *  and w in A such that (v,w) in Gd, v precedes w and, therefore, v
 *  will be scanned before w.  
 *  
 * We should note that A can be constructed from B in one depth-first
 * search computation.  It uses stacks to implement A and B. The
 * computation pops nodes from B one by one. For each removed node v
 * if v was already visited by the current depth-first search, we do
 * nothing. If v was not visited and v has no outgoing arcs of
 * negative reduced cost, we mark v as scanned.  If v was not visited
 * and has an outgoing arc of negative reduced cost, we visit in the
 * depth first order all nodes reachable from v in Gd and not visited
 * previously. At the end of the depth-first search visit of a node,
 * the node is marked labeled and pushed on A. Note that this
 * procedure can be implemented so that it looks only at the nodes
 * which were on B at the beginning of the computation or are on A at
 * the end and each such node is examined exactly once.
 * 
 * Now suppose G has cycles of zero or negative length. In this case
 * Gd need not be acyclic. If Gd has a negative length cycle, we can
 * terminate the computation. The zero length cycles in Gd can be
 * contracted.  However, we are going to use the same simplification
 * proposed and implemented by Cherkassky, Goldberg and
 * Radzik. Instead of contracting zero length cycles, we simply ignore
 * the backs arcs discovered during the depth-first search. The
 * resulting topological order is in the admissible graph minus the
 * ignored arcs. This change does not affect the algorithm correctness
 * or its running time bound.
 */

#include "spalgo.h"
#include "graph.h"
#include "tsnode.h"

#include <stack>

class topological_scan :
  public shortest_path_algorithm<topological_scan, graph<tsnode> > {
  /** Convenience type aliases.  */
  typedef shortest_path_algorithm<topological_scan, graph<tsnode> > inherited;
  typedef std::vector<node_t*> queue_t;

  /** Stacks of nodes to scan in the current pass and in the next pass.  */
  queue_t to_scan, to_scan_next;
  
  /** Inserts the node in the queue for the current pass.  */
  void push_this(node_t& node) {
    to_scan.push_back(&node);
  }

  /** Inserts the node in the queue for the next pass.  */
  void push(node_t& node) {
    to_scan_next.push_back(&node);
  }

  /** Returns true if there are not more nodes left to scan in the
      current pass.  */
  bool empty() const {
    return to_scan.empty();
  }

  /** Return true if the queue for the next pass is empty.  */
  bool next_empty() const {
    return to_scan_next.empty();
  }

  /** Removes a node from the top of the current-pass queue.  Make
      sure it is marked as not visited.  */
  node_t *pop() {
    node_t *node = to_scan.back();
    to_scan.pop_back();
    node->unvisit();
    return node;
  }

  /** Removes a node from the front of the next-pass queue.  */
  node_t *pop_next() {
    node_t *node = to_scan_next.back();
    to_scan_next.pop_back();
    return node;
  }

  /** Visit the given node.  Since the behavior for nodes removed from
      the next-pass queue is different from those that are just
      reached by them, the `from_next' flag is used to select the
      appropriate behavior.

      There are various admittedly strange decisions in this code,
      that we have taken for the sake of duplicating the algorithm of
      the original implementation by Cherkassky, Goldberg and Radzik.
      We'll try to clean up these apparent mistakes in another
      implementation. */
  void visit(node_t& node, bool from_next = false) {
    if (node.visit())
      return; // do not visit a node more than once in a pass

    /** Becomes true if we find any child in the admissible graph.  */
    bool found = false; 

    /** Was this node reached already? */
    bool reached = node.is_reached();

    adjlist_t &adjlist = node.adjlist(); // short-hand for adjacency list

    if (reached) {
      // Since we have been reached, our distance is well-defined
      distance_t dist = node.distance();
      
      // Scan all arcs in the current node.
      for(arc_iterator arc_ptr = adjlist.begin(), end_arc = adjlist.end();
	  arc_ptr != end_arc; ++arc_ptr) {
	arc_t& arc = *arc_ptr; // short-hand for the arc
	node_t& to = arc.to(); // the head of the arc
	if (!to.is_reached() // a path is always better than no path
	    || dist + arc.length() <= to.distance()) {
	  // even though the == part does lead to real improvement,
	  // we're taking it because the original algorithm does too
	  found = true;
	  visit(to); // now scan the child node
	}
      }
    } else {
      // It doesn't seem to make much sense to perform this scan,
      // since we can't really tell which distance is better: there
      // are no well-defined distances.  However, the original
      // algorithm doesn't deal with undefined distances, so we try to
      // duplicate the apparently broken behavior here
      for(arc_iterator arc_ptr = adjlist.begin(), end_arc = adjlist.end();
	  arc_ptr != end_arc; ++arc_ptr) {
	arc_t& arc = *arc_ptr; // short-hand for the arc
	node_t& to = arc.to(); // shoft-hand for the node
	if (!to.is_reached() && arc.length() <= 0) {
	  // in the original implementation, if the head was not
	  // reached and the arc has negative cost, it belongs to the
	  // admissible graph.  Strange, eh? :-)
	  found = true;
	  visit(to); // now scan the child node
	}
      }
    }

    if (found || !from_next || !reached) {
      node.mark_labeled(); // even if it is unreached, mark it as labeled
      push_this(node); // put it in the queue for this pass
      node.visited(); // mark it as visited

      // Note that we're ticking the counter when we push the node
      // into the queue, and then we tick it again after scanning it
      // in the main loop.  Yes, it is strange, but it's exactly what
      // the original implementation did.
      scan_tick();
    } else {
      // If we get here, no arc could be found in the admissible graph.
      scanned(node); // mark the node as scanned
      node.unvisit(); // and remove the visit mark
      // This is no good, since we might get back to this node.
      // Moreover, we're not counting this scan.  But then, the
      // original implementation isn't either.
    }
  }
      
public:
  /** Initialize the queue with the source node.  Note that it is the
      next pass queue that is initialized, so the algorithm starts
      with a topological scan.  Isn't this wasteful?  */
  topological_scan(graph_t& g) : inherited(g) {
    push(g.source()); // scan source first
  }

  /** Pick a node from the current-pass queue.  If the queue was
      empty, run the topological sorting heuristics to set up the next
      pass.  */
  node_t *select() {
    for(;;) {
      if (!empty())
	return pop(); // ok, we found a node
    
      if (next_empty())
	return 0; // job done, no more nodes to process
    
      do { // since next cannot empty here, we need not test it again
	node_t& node = *pop_next(); // pick a node from the next queue
	visit(node, true); // and visit it
      } while (!next_empty()); // until we have visited them all
    } // now restart
  }

  /** Return true iff the node was relabeled.  If it has become
      labeled, it is inserted in the queue.  */
  bool test_relabel(arc_t& arc) {
    bool was_labeled = arc.to().is_labeled();
    bool relabeled = inherited::test_relabel(arc);
    // was_labeled implies it was in the queue already
    if (relabeled && !was_labeled)
      push(arc.to()); // scan it in the next pass
    return relabeled;
  }
};

typedef topological_scan algorithm_t;

#include "main.h"
