// $Id: bfp.cc,v 1.6 1998/11/29 18:38:45 islene Exp $

/* Copyright 1998 Alexandre Oliva <oliva@dcc.unicamp.br>, Islene Calciolari Garcia <islene@dcc.unicamp.br>
 *
 * This file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * Parent-Checking (BFP): This algorithm is a heurist
 * [CherkasskyGoldbergRadzik93] applied to the Bellman-Ford-Moore
 * algorithm, based on the following observation. Suppose a node v has
 * just been removed from the queue and its parent, u = parent(v), is
 * in the queue. Note that d(v) was last updated when u was scanned
 * and d(v) was set to d(u) + l(u,v), and after that d(u) decreased
 * (causing u to be again added to the queue). Intuitively, is
 * wasteful to scan v at this point because we know that d(v) will
 * decrease. The BFP algorithm scans a node only if its parent is not
 * in the queue, and it maintains the O(mn) running time.
 */

#include "spalgo.h"
#include "graph.h"
#include "node.h"
#include "basic_node.h"

#include <queue>

class parent_checking : 
  public shortest_path_algorithm<parent_checking, graph< basic_node > > {
  /** Convenience type alias.  */
  typedef shortest_path_algorithm<parent_checking, graph< basic_node > > inherited;

  /** Queue of nodes to scan.  */
  std::queue<node_t*> to_scan;
  
  /** Inserts the node in the back of the queue.  */
  void push(node_t& node) {
    to_scan.push(&node);
  }

  /** Returns true if there are no more nodes left to scan.  */
  bool empty() const {
    return to_scan.empty();
  }

  /** Removes a node from the front of the queue.  */
  node_t *pop() {
    node_t *node = to_scan.front();
    to_scan.pop();
    return node;
  }

public:
  /** Initialize the queue with the source node.  */
  parent_checking(graph_t& g) : inherited(g) {
    push(g.source()); // scan source first
  }

  /** If the parent of the front node is in the queue, mark the node
      as scanned and pick the next node.  */
  node_t *select() {
    if (empty())
      return 0;
    // we no longer have to check for emptiness: a node will only be
    // skipped if its parent is in the queue too
    for(;;) {
      node_t *node = pop(), *parent = node->parent_node();
      // if parent is in the queue (i.e., labeled), skip it
      if (parent && parent->is_labeled())
	scanned(*node);
      else
	return node;
    }
  }

  /** Return true iff the node was relabeled.  If it has become
      labeled, it is inserted in the queue.  */
  bool test_relabel(arc_t& arc) {
    bool was_labeled = arc.to().is_labeled();
    bool relabeled = inherited::test_relabel(arc);
    // was_labeled implies it was in the queue already
    if (relabeled && !was_labeled)
      push(arc.to()); // scan it later
    return relabeled;
  }
};

typedef parent_checking algorithm_t;

#include "main.h"
