/* Expires e-mail from a Unix-format mailbox.
   Copyright (C) 1993-1995,1997,1999 Alexandre Oliva <oliva@dcc.unicamp.br>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#if HAVE_CONFIG_H
#include "config.h"
#endif

#if HAVE_SYS_FILE_H
#include <sys/file.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_UTIME_H
#include <utime.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

inline void perror2(char *p1, char *p2) {
  fprintf(stderr, "%s: ", p1);
  perror(p2);
}

inline int minimum(int a, int b) {
  return (a < b) ? a : b;
}

int timedout() {
  fprintf(stderr, "lock timed out\n");
  exit(1);
}

static const char *const dateformats[] = {
  "%C", /* locale's long-format date and time */
  "%a%b%d%T%Z%Y", /* day of week, month, day of month, time with
                     seconds, timezone, year */
  "%b%d%T%Z%Y", /* without day of week */
  "%a%b%d%R%Z%Y", /* day of week, month, day of month, time without
                     seconds, year */
  "%b%d%R%Z%Y", /* without day of week */
  "%a%b%d%T%Y", /* without timezone */
  "%b%d%T%Y", /* without timezone and day of week */
  "%a%b%d%R%Y" /* without timezone and seconds */
  "%b%d%R%Y" /* without timezone, seconds and day of week */
};

int main(int argc, char *argv[]) {
  int retval = 0;
  char *ifile, *command;
  int days, minsize;
  time_t curtime, endtime, msgtime;
  struct stat filestat;
  struct utimbuf tms;
  FILE *ifl;
  char *header;
  int ch;
  size_t pos;
  size_t lastpos;

  if (argc != 5) {
    fprintf(stderr, "usage: %s <mail-file> <days> <minsize> <command>\n\
%s reads the mail file and pipes all messages older than <days> days to command, if they are, together, taking more than <minsize> bytes. If the command returns 0, the messages are removed from the original file.\n", argv[0], argv[0]);
    return 1;
  }

  ifile = argv[1];
  days = atoi(argv[2]);
  minsize = atoi(argv[3]);
  command = argv[4];
  curtime = time(NULL);
  endtime = mktime(gmtime(&curtime)) - 24*3600*days;
  msgtime = 0;

  if (stat(ifile, &filestat) == -1) {
    perror2(argv[0], ifile);
    return errno;
  }
    
  {
    ifl = fopen(ifile, "r+");
    if (ifl == NULL) {
      perror2(argv[0], ifile);
      return errno;
    }

    signal(SIGALRM, timedout);
    alarm(30);

    if (lockf(fileno(ifl), F_LOCK, 0) == -1) {
      perror2(argv[0], ifile);
      goto closeandfail;
    }

    alarm(0);

    {
      header = "\n\nFrom ";
      pos = 2;
      lastpos = ftell(ifl);

      if (lastpos == -1) {
	perror2(argv[0], ifile);
	goto errorwentup;
      }

      while (msgtime < endtime) {
	if ((ch = fgetc(ifl)) == EOF) {
	  lastpos = ftell(ifl);
	  if (lastpos == -1) {
	    perror2(argv[0], ifile);
	    goto errorwentup;
	  }
	  break;
	} else if (ch == header[pos]) {
	  if (header[++pos] == 0) {
	    char date[512];
	    struct tm tmdate;
	    char *start, *end;

	    lastpos = ftell(ifl);
	    if (lastpos == -1) {
	      perror2(argv[0], ifile);
	      goto errorwentup;
	    }
	    pos = 0;
	    lastpos -= strlen(header+2);

	    if (fgets(date, sizeof(date), ifl) != date)
	      break;

	    if (feof(ifl))
	      break;

	    start = date;
	    do {
	      int i;

	      end = 0;
	      for (i = 0;
		   end == 0 && i < sizeof(dateformats)/sizeof(*dateformats);
		   ++i)
		end = strptime(start, dateformats[i], &tmdate);
	      if (!end) {
		while (*start != ' ' && *start != '\n')
		  ++start;
		if (*start == '\n') {
		  fprintf(stderr, "%s: could not find date in string %s",
			  ifile, date);
		  break;
		}
		++start;
	      } else if (*end != '\n')
		break;
	    } while(!end);

	    msgtime = mktime(&tmdate);
	  }
	} else
	  pos = (ch == header[0]);
      }

      if (lastpos >= minsize) {
	char buffer[16384];
	const int buffersize = sizeof(buffer);
	FILE *output;
	int written;

	if (buffer == NULL) {
	  perror(argv[0]);
	  goto errorwentup;
	}

	if ((output = popen(command, "w")) == NULL) {
	  perror2(argv[0], command);
	  goto errorwentup;
	}

	rewind(ifl);

	written = 0;
	while (written < lastpos) {
	  int towrite = fread(buffer, 1,
			      minimum(lastpos - written, buffersize),
			      ifl);
	  int nwritten = 0;

	  while (nwritten < towrite) {
	    int thiswrite = fwrite(buffer+nwritten, 1, towrite-nwritten, output);
	    if (thiswrite != towrite-nwritten) {
	      perror2(argv[0], ifile);
	      goto closeandfail;
	    }
	    nwritten += thiswrite;
	  }

	  written += nwritten;
	}

	if (pclose(output) != 0)
	  fprintf(stderr,
		  "%s: %s did not return 0. Will not remove messages!\n",
		  argv[0], command);
	else {
	  output = tmpfile();
	  if (output == NULL)
	    goto errorwentup;

	  while (!feof(ifl)) {
	    int towrite = fread(buffer, 1, buffersize, ifl);
	    int nwritten = 0;

	    while (nwritten < towrite)
	      nwritten += fwrite(buffer+nwritten, 1, towrite-nwritten,
				 output);
	  }

	  if (ftruncate(fileno(ifl), ftell(output)))
	    perror2(argv[0], ifile);
	  else {
	    rewind(output);
	    rewind(ifl);

	    while (!feof(output)) {
	      int towrite = fread(buffer, 1, buffersize, output);
	      int nwritten = 0;

	      while (nwritten < towrite)
		nwritten += fwrite(buffer+nwritten, 1, towrite-nwritten,
				   ifl);
	    }
	  }
	}
      }
    }

    if (0) { closeandfail: retval = errno; }
    
    if (lockf(fileno(ifl), F_ULOCK, 0) == -1) {
      perror2(argv[0], ifile);
      goto errorwentup;
    }

    if (fclose(ifl) != 0) {
      perror2(argv[0], ifile);
      goto errorwentup;
    }
  }
  
  if (0) { errorwentup: retval = errno; }
  
  tms.actime = filestat.st_atime;
  tms.modtime = filestat.st_mtime;
  if (utime(ifile, &tms) != 0) {
    perror2(argv[0], ifile);
    return errno;
  }

  return retval;
}
